/******************************************************************************
 *
 * Copyright (c) 1996-1999 Palm, Inc. or its subsidiaries.
 * All rights reserved.
 *
 * File: MailDB.h
 *
 * Release: Palm OS SDK 4.0 (63220)
 *
 * Description:
 *	  This file defines the database structures and functions of the Mail
 *   application.
 *
 * History:
 *		May 30, 1996	Created by Art Lamb
 *
 *****************************************************************************/

/************************************************************
 *
 * Mail database constants.
 *
 *************************************************************/

// String resource used to initialize application info block.
#define mailLocalizedAppInfoStr	1000

// Dirty flags for to do application info
#define mailSortByDirty	0x0001


// Sort orders
#define sortByDate					0
#define sortByFrom					1
#define sortBySubject				2


#define noUnsendMessage				-1

/************************************************************
 *
 * Mail database structures.
 *
 *************************************************************/

// The format of a packed mail record is as follows:
//
//	DateType			 		date;
//	TimeType			 		time;
//	MailDBRecordFlags		flags;
//	char []					subject;			// null terminated
//	char []					from;				// null terminated
//	char []					to;				// null terminated
//	char []					cc;				// null terminated
//	char []					bcc;				// null terminated
//	char []					replyTo;			// null terminated
// char []					sentTo;			// null terminated
//	char []					body;				// null terminated


// Application info block 
typedef struct {
   UInt16			renamedCategories;	// bitfield of categories with a different name
	char 				categoryLabels [dmRecNumCategories] [dmCategoryLength];
	UInt8 			categoryUniqIDs [dmRecNumCategories];
	UInt8				lastUniqID;				// Uniq IDs generated by the device are between
													// 0 - 127.  Those from the PC are 128 - 255.
	UInt8				reserved1;				// from the compiler word aligning things
	UInt16			reserved2;
	UInt16			dirtyAppInfo;
	UInt8				sortOrder;
	UInt8				reserved3;
	UInt32			unsentMessage;			// unique id of unsend message.
	UInt16			signatureOffset;
} MailAppInfoType;

typedef MailAppInfoType * MailAppInfoPtr;

typedef enum { priorityHigh, priorityNormal, priorityLow } MailPriorityType;

typedef enum { sentTo, sentCC, sentBCC } MailAddressingType;

typedef struct {
	UInt16				read					: 1;
	UInt16				signature			: 1;
	UInt16				confirmRead			: 1;
	UInt16				confirmDelivery	: 1;
	UInt16				priority				: 2;		// MailPriorityType
	UInt16				addressing			: 2;		// MailAddressingType
	UInt16				reserved				: 8;
} MailFlagsType;


// The following structure doesn't really exist.  The first field
// varies depending on the data present.  However, it is convient
// (and less error prone) to use when accessing the other information.
typedef struct {
	DateType			 		date;
	TimeType					time;
	MailFlagsType			flags;
	Char						firstField;
	UInt8						reserved;
} MailPackedDBRecordType;

typedef MailPackedDBRecordType * MailPackedDBRecordPtr;


// MailItemType
//
// This is the format of a to do record used by the application.  All 
// pointers are either NULL or point to data within the PackedDB record.
// All strings are null character terminated.
typedef struct {
	DateType 				date;
	TimeType					time;
	MailFlagsType			flags;
	Char *					subject;
	Char *					from;
	Char *					to;
	Char *					cc;
	Char *					bcc;
	Char *					replyTo;
	Char *					sentTo;
	Char *					body;
} MailDBRecordType;

typedef MailDBRecordType * MailDBRecordPtr;



//This structure should match the order of the fields in the record.
typedef enum { 
	mailFieldDate,
	mailFieldTime,
	mailFieldFlags,
	mailFieldSubject,
	mailFieldFrom,
	mailFieldTo,
	mailFieldCC,
	mailFieldBCC,
	mailFieldReplyTo,
	mailFieldSentTo,
	mailFieldBody } MailRecordFieldType;


// Used by MailChangeRecord function.
typedef struct {
	unsigned 	date		:1;
	unsigned 	time		:1;
	unsigned		flags		:1;
	unsigned		subject	:1;
	unsigned		from		:1;
	unsigned		to			:1;
	unsigned		cc			:1;
	unsigned		bcc		:1;
	unsigned		replyTo	:1;
	unsigned		sentTo	:1;
	unsigned		body		:1;
} MailChangedFieldsType;




/************************************************************
 *
 * Mail database routines.
 *
 *************************************************************/
#ifdef __cplusplus
extern "C" {
#endif


Err			MailAppInfoInit (DmOpenRef dbP);

MemHandle	MailGetAppInfo (DmOpenRef dbP);

UInt8 		MailGetSortOrder (DmOpenRef dbP);

Err			MailChangeSortOrder (DmOpenRef dbP, UInt8 sortOrder);

void 			MailSort (DmOpenRef dbP);

UInt32		MailGetUnsentMessage (DmOpenRef dbP);

void			MailSetUnsentMessage (DmOpenRef dbP, UInt32 uniqueID);

MemHandle 	MailGetSignature (DmOpenRef dbP);

void 			MailSetSignature (DmOpenRef dbP, MemHandle h);



void 			MailParseAddress (Char * address, Boolean realName, Char * buffer,
					UInt16 bufferLen);
				
Err 			MailNewRecord (DmOpenRef dbP, MailDBRecordPtr r, UInt16 *indexP);

Err			MailAttachRecord (DmOpenRef dbP, MemHandle recordH, UInt16 * indexP);

Err 			MailGetRecord (DmOpenRef dbP, UInt16 index, MailDBRecordPtr r,
					MemHandle * handleP);

Err 			MailChangeRecord (DmOpenRef dbP, UInt16 * index, 
					MailDBRecordPtr r, MailChangedFieldsType changedFields);

void			MailUnpackRecord (MailPackedDBRecordPtr src, MailDBRecordPtr dest);

Err 			MailChangeRecordField (DmOpenRef dbP, UInt16 * indexP, 
					MailRecordFieldType recordField, void * data);

Err 			MailInsertIntoField (DmOpenRef dbP, UInt16 index, 
					MailRecordFieldType recordField, Char * insertData, 
					UInt16 insertLen, UInt16 insertOffset);

Err 			MailSortRecord (DmOpenRef dbP, UInt16 * indexP);


void 			ECMailDBValidate (DmOpenRef dbP);


Err 			MailGetDatabase (DmOpenRef *dbPP, UInt16 mode);



#ifdef __cplusplus
}
#endif
